program NEWTON2DIM;
{--------------------------------------------------------------------}
{  Alg2'10.pas   Pascal program for implementing Algorithm 2.10      }
{                                                                    }
{  NUMERICAL METHODS: Pascal Programs, (c) John H. Mathews 1995      }
{  To accompany the text:                                            }
{  NUMERICAL METHODS for Math., Science & Engineering, 2nd Ed, 1992  }
{  Prentice Hall, Englewood Cliffs, New Jersey, 07632, U.S.A.        }
{  Prentice Hall, Inc.; USA, Canada, Mexico ISBN 0-13-624990-6       }
{  Prentice Hall, International Editions:   ISBN 0-13-625047-5       }
{  This free software is compliments of the author.                  }
{  E-mail address:       in%"mathews@fullerton.edu"                  }
{                                                                    }
{  Algorithm 2.10 (Newton-Raphson Method in 2-Dimensions).           }
{  Section   2.7, Newton's Method for Systems, Page 116              }
{--------------------------------------------------------------------}

  uses
    crt;

  const
    Max = 99;
    Rmax = 101;
    Del = 1E-6;
    Eps = 1E-6;
    FunMax = 5;

  type
    MATRIX = array[1..2, 1..2] of real;
    VECTOR = array[1..2] of real;
    RVECTR = array[0..Rmax] of real;
    LETTERS = string[200];
    STATUS = (Computing, Done, Working);

  var
    Cond, FunType, Inum, K, Sub: integer;
    Delta, Epsilon, Rnum: real;
    Pstart, P0, P1, F1: VECTOR;
    VP, VQ: RVECTR;
    Ans: CHAR;
    Mess: LETTERS;
    Stat, State: STATUS;

  procedure FUN (P: VECTOR; var F: VECTOR);
    var
      X, Y: real;
  begin
    X := P[1];
    Y := P[2];
    case FunType of
      1: 
        begin
          F[1] := X * X - 2 * X - Y + 0.5;
          F[2] := X * X + 4 * Y * Y - 4;
        end;
      2: 
        begin
          F[1] := X * X + Y * Y - 2;
          F[2] := X * X - Y;
        end;
      3: 
        begin
          F[1] := X * X + Y * Y - 2;
          F[2] := X * Y - 1;
        end;
      4: 
        begin
          F[1] := 2 * X * Y - 3;
          F[2] := X * X - Y - 2;
        end;
      5: 
        begin
          F[1] := 3 * X * X - 2 * Y * Y - 1;
          F[2] := X * X - 2 * X + Y * Y + 2 * Y - 8;
        end;
    end;
  end;

  procedure JACOBIAN (P: VECTOR; var D: MATRIX);
    var
      X, Y: real;
  begin
    case FunType of
      1: 
        begin
          X := P[1];
          Y := P[2];
          D[1, 1] := 2 * X - 2;                                         {dF1/dx}
          D[1, 2] := -1;                                            {dF1/dy}
          D[2, 1] := 2 * X;                                           {dF2/dx}
          D[2, 2] := 8 * Y;                                           {dF2/dy}
        end;
      2: 
        begin
          X := P[1];
          Y := P[2];
          D[1, 1] := 2 * X;
          D[1, 2] := 2 * Y;
          D[2, 1] := 2 * X;
          D[2, 2] := -1;
        end;
      3: 
        begin
          X := P[1];
          Y := P[2];
          D[1, 1] := 2 * X;
          D[1, 2] := 2 * Y;
          D[2, 1] := Y;
          D[2, 2] := X;
        end;
      4: 
        begin
          X := P[1];
          Y := P[2];
          D[1, 1] := 2 * Y;
          D[1, 2] := 2 * X;
          D[2, 1] := 2 * X;
          D[2, 2] := -1;
        end;
      5: 
        begin
          X := P[1];
          Y := P[2];
          D[1, 1] := 6 * X;
          D[1, 2] := -4 * Y;
          D[2, 1] := 2 * X - 2;
          D[2, 2] := 2 * Y + 2;
        end;
    end;
  end;

  procedure PRINTFUNCTIONS (FunType: integer);
  begin
    case FunType of
      1: 
        begin
          WRITELN('0 = f (x,y) = x^2 - 2x - y + 0.5');
          WRITELN('                    1');
          WRITELN('               0 = f (x,y) = x^2 + 4 y^2 - 4');
          WRITELN('                    2');
        end;
      2: 
        begin
          WRITELN('0 = f (x,y) = x^2 + y^2 - 2');
          WRITELN('                    1');
          WRITELN('               0 = f (x,y) = x^2 - y');
          WRITELN('                    2');
        end;
      3: 
        begin
          WRITELN('0 = f (x,y) = x^2 + y^2 - 2');
          WRITELN('                    1');
          WRITELN('               0 = f (x,y) = xy - 1');
          WRITELN('                    2');
        end;
      4: 
        begin
          WRITELN('0 = f (x,y) = 2xy - 3');
          WRITELN('                    1');
          WRITELN('               0 = f (x,y) = x^2 - y - 2');
          WRITELN('                    2');
        end;
      5: 
        begin
          WRITELN('0 = f (x,y) = 3 x^2 - 2 y^2 - 1');
          WRITELN('                    1');
          WRITELN('               0 = f (x,y) = x^2 - 2x + y^2 + 2y - 8');
          WRITELN('                    2');
        end;
    end;
  end;

  procedure NEWTON (var P0, P1: VECTOR; var F1: VECTOR; var Cond, K: integer; Delta, Epsilon: real; Max: integer);
    var
      DET: real;
      FnZero, Small, RelErr: real;
      D: MATRIX;
      DP, F0: VECTOR;
  begin
    Small := Delta;
    K := 0;
    Cond := 0;
    VP[0] := P0[1];  {Store an array of answers}
    VQ[0] := P0[2];
    FUN(P0, F0);
    P1 := P0;                                        {Vector replacement}
    F1 := F0;                                        {Vector replacement}
    while (K < Max) and (Cond = 0) do
      begin
        P0 := P1;                                    {Vector replacement}
        F0 := F1;                                    {Vector replacement}
        K := K + 1;
        JACOBIAN(P0, D);
        DET := D[1, 1] * D[2, 2] - D[1, 2] * D[2, 1];
        if DET = 0 then
          begin
            DP[1] := 0;
            DP[2] := 0;
            Cond := 1;
          end
        else
          begin
            DP[1] := (F0[1] * D[2, 2] - F0[2] * D[1, 2]) / DET;
            DP[2] := (F0[2] * D[1, 1] - F0[1] * D[2, 1]) / DET;
          end;
        P1[1] := P0[1] - DP[1];
        P1[2] := P0[2] - DP[2];
        VP[K] := P1[1];  {Store an array of answers}
        VQ[K] := P1[2];
        FUN(P1, F1);
        RelErr := (ABS(DP[1]) + ABS(DP[2])) / (ABS(P1[1]) + ABS(P1[2]) + Small);
        FnZero := ABS(F1[1]) + ABS(F1[2]);
        if (RelErr < Delta) and (FnZero < Epsilon) then
          if Cond <> 1 then
            Cond := 2;
      end;
  end;

  procedure INPUTTOL (var Delta, Epsilon: real);
  begin
    CLRSCR;
    WRITELN('                         ->                          ->');
    WRITELN('          Starting with  P   a sequences of points { P  } is generated');
    WRITELN('                          0                           K');
    WRITELN;
    WRITELN('     which converges to the solution.  Newton`s method uses the formulas:');
    WRITELN;
    WRITELN('                 ->                -1  ->     ->       ');
    WRITELN('                 P     =   P   -  J  ( P  ) * F ( P  ).');
    WRITELN('                  K+1       K           K          K   ');
    WRITELN;
    WRITELN('     Termination occurs when');
    WRITELN;
    WRITELN;
    WRITELN('                 ->   ->                     -> ->             ');
    WRITELN('                |P  - P   | < Delta   AND   |F (P )| < Epsilon.');
    WRITELN('                  N    N-1                       N             ');
    WRITELN;
    WRITELN;
    Mess := '     Now  ENTER  the  value   Delta = ';
    Delta := Del;
    WRITE(Mess);
    READLN(Delta);
    Delta := ABS(Delta);
    if Delta < Del then
      Delta := Del;
    Mess := '     Now  ENTER  the value  Epsilon = ';
    Epsilon := Eps;
    WRITE(Mess);
    READLN(Epsilon);
    Epsilon := ABS(Epsilon);
    if Epsilon < Eps then
      Epsilon := Eps;
  end;                                 {End of procedure INPUTTOL}

  procedure MESSAGE (var Delta, Epsilon: real);
  begin
    CLRSCR;
    WRITELN;
    WRITELN;
    WRITELN;
    WRITELN('                    NEWTON`S METHOD FOR SYSTEMS');
    WRITELN;
    WRITELN;
    WRITELN;
    WRITELN('    Newton`s method is used to find the solution of the nonlinear system');
    WRITELN;
    WRITELN;
    WRITELN('                            0  =  f (x,y)');
    WRITELN('                                   1');
    WRITELN;
    WRITELN('                            0  =  f (x,y).');
    WRITELN('                                   2');
    WRITELN;
    WRITELN;
    WRITELN;
    WRITE('                         Press the <ENTER> key.  ');
    READLN(Ans);
    WRITELN;
    WRITELN('                              ->');
    WRITELN('                Suppose that  P   has been obtained.  Then the');
    WRITELN('                               K');
    WRITELN;
    WRITELN('                    ->  ->                         ->');
    WRITELN('          function  F ( P  )  and the Jacobian  J( P  )  are computed.');
    WRITELN('                         K                          K');
    WRITELN;
    WRITELN('                                       ->   -->       ->  ->  ');
    WRITELN('          Next, the linear system   J( P  ) dP   =  - F ( P  )');
    WRITELN('                                        K                  K  ');
    WRITELN;
    WRITELN('                                    -->');
    WRITELN('          is solved, and the value  dP   is determined.');
    WRITELN;
    WRITELN;
    WRITELN('                                ->        ->       -->  ');
    WRITELN('          The next iterate is   P     =   P    +   dP .');
    WRITELN('                                 K+1       K         K  ');
    WRITELN;
    WRITELN;
    WRITELN;
    WRITE('                           Press the <ENTER> key.  ');
    READLN(Ans);
    WRITELN;
    INPUTTOL(Delta, Epsilon);
  end;                                  {End of procedure MESSAGE}

  procedure GETFUN (var FunType: integer);
    var
      K: integer;
  begin
    CLRSCR;
    WRITELN('               Choose your system of functions:');
    WRITELN;
    for K := 1 to FunMax do
      begin
        WRITE('          <', K : 1, '>  ');
        PRINTFUNCTIONS(K);
      end;
    Mess := '               SELECT < 1 - 5 >  ';
    FunType := 1;
    WRITE(Mess);
    READLN(FunType);
    if FunType < 1 then
      FunType := 1;
    if FunType > FunMax then
      FunType := FunMaX;
  end;

  procedure GETPOINTS (var Pstart, P0: VECTOR);
  begin
     {CLRSCR;
    WRITELN('     You chose to solve the non-linear system:');
    WRITELN;
    WRITE('               ');
    PRINTFUNCTIONS(FunType);
    WRITELN;
    WRITELN;
    WRITELN('     Enter the initial starting point  (p ,q )');
    WRITELN('                                         0  0');
    Mess := '     You must now ENTER the coordinate  p0 = ';
    P0[1] := 0;
    WRITE(Mess);
    READLN(P0[1]);
    Mess := '     You must now ENTER the coordinate  q0 = ';
    P0[2] := 0;
    WRITE(Mess);
    READLN(P0[2]);
    Pstart := P0;   {Vector replacement}
  end;

  procedure RESULTS (Pstart, P0, P1: VECTOR; F1: VECTOR; Cond, K: integer);
  begin
    CLRSCR;
    WRITELN;
    WRITELN('Newton-Raphson iteration was used to solve the non-linear system:');
    WRITELN;
    WRITE('               ');
    PRINTFUNCTIONS(FunType);
    WRITELN;
    WRITELN('starting with  p  =', Pstart[1] : 15 : 7, '  and   q  =', Pstart[2] : 15 : 7);
    WRITELN('                0                             0');
    WRITELN;
    WRITELN('It took ', K : 2, ' iterations to compute the solution point (P,Q).');
    WRITELN;
    case Cond of
      0: 
        WRITELN('However, the maximum number of iterations was exceeded.');
      1: 
        WRITELN('However, division by zero was encountered.');
      2: 
        WRITELN('The solution is within the desired tolerances.');
    end;
    WRITELN;
    case Cond of
      0, 1: 
        WRITELN('         P(', K : 2, ') =', P1[1] : 15 : 7, '  ,  Q(', K : 2, ') =', P1[2] : 15 : 7);
      2: 
        WRITELN('            P  =', P1[1] : 15 : 7, '  ,     Q  =', P1[2] : 15 : 7);
    end;
    WRITELN;
    WRITELN('          |DP| =', ABS(P1[1] - P0[1]) : 15 : 7, '  ,   |DQ| =', ABS(P1[2] - P0[2]) : 15 : 7);
    WRITELN;
    WRITELN('       f (P,Q) =', F1[1] : 15 : 7);
    WRITELN('        1');
    WRITELN('       f (P,Q) =', F1[2] : 15 : 7);
    WRITELN('        2');
  end;

  procedure PRINTAPPROXS;
    var
      J: integer;
  begin
    CLRSCR;
    WRITELN;
    WRITELN('           k             p                      q  ');
    WRITELN('                          k                      k ');
    WRITELN('         ---------------------------------------------------');
    WRITELN;
    for J := 0 to K do
      begin
        WRITELN('          ', J : 2, '     ', VP[J] : 15 : 7, '     ', VQ[J] : 15 : 7);
        WRITELN;
        if J mod 11 = 9 then
          begin
            WRITE('                  Press the <ENTER> key.  ');
            READLN(Ans);
            WRITELN;
          end;
      end;
    WRITELN;
    WRITE('                  Press the <ENTER> key.  ');
    READLN(Ans);
  end;

begin                                            {Begin Main Program}
  MESSAGE(Delta, Epsilon);
  Stat := Working;
  while (Stat = Working) do
    begin
      GETFUN(FunType);
      State := Computing;
      while (State = Computing) do
        begin
          GETPOINTS(Pstart, P0);
          NEWTON(P0, P1, F1, Cond, K, Delta, Epsilon, Max);
          RESULTS(Pstart, P0, P1, F1, Cond, K);
          WRITELN;
          WRITE('Want  to see  all  of the approximations ?  <Y/N>  ');
          READLN(Ans);
          WRITELN;
          if (Ans = 'Y') or (Ans = 'y') then
            PRINTAPPROXS;
          WRITELN;
          WRITE('Want to try  a different  starting point ?  <Y/N>  ');
          READLN(Ans);
          WRITELN;
          if (Ans <> 'Y') and (Ans <> 'y') then
            State := Done;
          if (Ans = 'Y') or (Ans = 'y') then
            CLRSCR;
        end;
      WRITELN;
      WRITE('Want to try a different nonlinear system ?  <Y/N>  ');
      READLN(Ans);
      WRITELN;
      if (Ans <> 'Y') and (Ans <> 'y') then
        Stat := Done
    end;
end.                                               {End Main Program}

